/*
 * Android development
 * crestron_audio.c -UIHal for crestron Audio on the tst600 
 *
 * Written: Ray Kuschan
 *
 *  This module is the UI HAL abstraction layer calls into the audio driver.
 */

#include <linux/module.h>
#include <linux/i2c.h>
#include <linux/slab.h>
#include <linux/interrupt.h>
#include <linux/input.h>
#include <linux/delay.h>
#include <linux/signal.h>
#include <asm/siginfo.h>	//siginfo
#include <linux/debugfs.h>
#include <sound/core.h>
#include <sound/pcm.h>
#include <sound/pcm_params.h>
#include <sound/soc.h>
#include <sound/soc-dapm.h>
#include <sound/initval.h>

#include "../tlv320aic32x4.h"
#include "crest_audio.h"

#include <linux/proc_fs.h>  /* Necessary because we use proc fs */
#include <asm/uaccess.h>    /* for copy_*_user */

/* externs */
extern u8 aic32x4_read (struct snd_soc_codec *codec, u16 reg);
extern u16 aic32x4_read_2byte (struct snd_soc_codec * codec, u16 reg);
extern int aic32x4_reset_cache (struct snd_soc_codec *codec);
//extern int aic32x4_change_page(struct snd_soc_codec *codec, u8 new_page); //shouldnt ever need to change page since page contained in reg address
extern int aic32x4_write(struct snd_soc_codec *codec, u16 reg, u8 value);
extern void aic32x4_write_reg_cache(struct snd_soc_codec *codec,
                                    u16 reg, u8 value);
/* UI externs */
int aic32x4_mute(struct snd_soc_dai *dai, int mute);
extern int ui_aic32x4_read (struct snd_soc_codec *codec, u16 reg);
extern int ui_aic32x4_write (struct snd_soc_codec *codec, u8 param, u32 value);


extern struct uiAudio *uiAudio_data;
	
struct snd_soc_codec *gCodec;

int uiAudioInit(struct snd_soc_codec *codec);

#define UIAUDIO_FILENAME           "crestAudio"
#define PROCFS_MAX_SIZE     2048


#if 0
static unsigned int gSignalPID = 0;
#endif

/** * The structure keeping information about the /proc file */
static struct proc_dir_entry *UiAudio_File;

/** * The buffer (2k) for this module */
static char gAudio_buffer[PROCFS_MAX_SIZE];

/** * The size of the data held in the buffer */
static unsigned long gAudio_buffer_size = 0;

/* TBD: need to set a preset AGC value */
static int ui_agcRecEnable(struct snd_soc_dai *dai, int val)
{	
	return 0;
}
/* TBD: need to set a preset DRC value */
static int ui_agcPlayEnable(struct snd_soc_dai *dai, int val)
{	
	return 0;
}

/* write Crestron UI defaults to hardware */
int ui_AudioSetDefaults(struct snd_soc_dai *dai)
{
	mutex_lock(&gCodec->mutex);
	/* set play Vol */
	playVolume(dai);

	/* set Play AGC */
	ui_agcPlayEnable(dai, uiAudio_data->play_agc_capable);
	
	/* set Rec AGC */
	ui_agcRecEnable(dai, uiAudio_data->rec_agc_capable);

	/* enable Rec Boost level*/
	recBoostLevel(dai);

	mutex_unlock(&gCodec->mutex);

}

/* initialize the UI audio data structure */
int uiAudioDataDefInit(struct snd_soc_dai *dai)
{
	mutex_lock(&gCodec->mutex);
        uiAudio_data->play_capable=DEF_PLAY_CAP;   //translates to mute in DAC_MUTE_CTRL_REG
        uiAudio_data->rec_capable=DEF_REC_CAP;
        uiAudio_data->play_vol_default=DEF_PLAY_VOL;
        uiAudio_data->rec_boost_capable=DEF_BOOST_CAP;

        uiAudio_data->rec_boost_min=BOOST_MIN;     //boost range is (-100) - 100
        uiAudio_data->rec_boost_max=BOOST_MAX;

        uiAudio_data->aec_default=DEF_AEC_CAP;
        uiAudio_data->aec_capable=uiAudio_data->aec_default;

        uiAudio_data->play_agc_default=DEF_PLAY_AGC;
        uiAudio_data->play_agc_capable=uiAudio_data->play_agc_default;

        uiAudio_data->rec_agc_default=DEF_REC_AGC;
        uiAudio_data->rec_agc_capable=uiAudio_data->rec_agc_default;

	uiAudio_data->play_vol=uiAudio_data->play_vol_default;
	uiAudio_data->rec_boost_level=DEF_BOOST_LEVEL;
	uiAudio_data->play_mute=DEF_PLAY_MUTE;
	uiAudio_data->rec_mute=DEF_REC_MUTE;

	mutex_unlock(&gCodec->mutex);
	ui_AudioSetDefaults(dai);  //need to write the UI defaults to hardware
}
/** * This funtion is called when the /proc audio file is read */
static ssize_t procfs_audio_read(struct file *filp,	/* see include/linux/fs.h   */
			     char *buffer,	/* buffer to fill with data */
			     size_t length,	/* length of the buffer     */
			     loff_t * data)
{
  static int finished = 0;
  char * p = gAudio_buffer;
  //struct snd_soc_codec *codec=data;

  /* needed to stop from continuously printing */ 
  if ( finished == 1 ) { finished=0; return 0; }
  finished = 1;

   p += sprintf ( p, "PLAY_CAPABLE=%d\n" , uiAudio_data->play_capable );
   p += sprintf ( p, "REC_CAPABLE=%d\n" , uiAudio_data->rec_capable );
   p += sprintf ( p, "PLAY_VOL_DEFAULT=%d\n" , uiAudio_data->play_vol_default );
   p += sprintf ( p, "REC_BOOST_CAPABLE=%d\n" , uiAudio_data->rec_boost_capable );
   p += sprintf ( p, "REC_BOOST_MIN=%d\n" , uiAudio_data->rec_boost_min );
   p += sprintf ( p, "REC_BOOST_MAX=%d\n" , uiAudio_data->rec_boost_max );
   p += sprintf ( p, "AEC_CAPABLE=%d\n" , uiAudio_data->aec_capable );
   p += sprintf ( p, "AEC_DEFAULT=%d\n" , uiAudio_data->aec_capable );
   p += sprintf ( p, "PLAY_AGC_CAPABLE=%d\n" , uiAudio_data->play_agc_capable );
   p += sprintf ( p, "PLAY_AGC_DEFAULT=%d\n" , uiAudio_data->play_agc_default );
   p += sprintf ( p, "REC_AGC_CAPABLE=%d\n" , uiAudio_data->rec_agc_capable );
   p += sprintf ( p, "REC_AGC_DEFAULT=%d\n" , uiAudio_data->rec_agc_default );
   p += sprintf ( p, "PLAY_VOL=%d\n" , uiAudio_data->play_vol );
   p += sprintf ( p, "REC_BOOST_LEVEL=%d\n" , uiAudio_data->rec_boost_level );
   p += sprintf ( p, "PLAY_MUTE=%d\n" , uiAudio_data->play_mute );
   p += sprintf ( p, "REC_MUTE=%d\n" , uiAudio_data->rec_mute );

   gAudio_buffer_size = p-gAudio_buffer;

   if ( copy_to_user(buffer, gAudio_buffer, gAudio_buffer_size) ) {
	return -EFAULT;
   }

   return gAudio_buffer_size;
}

static ssize_t procfs_audio_write(struct file *file, const char *buffer, size_t len, loff_t * off)
{
	char* tag = NULL;
	char* value = NULL;
	char** tempPtr = &buffer;
	u8 regVal=0;

	gAudio_buffer_size = len;
	if (gAudio_buffer_size > PROCFS_MAX_SIZE ) {
		gAudio_buffer_size = PROCFS_MAX_SIZE;
	}

	if ( copy_from_user(gAudio_buffer, buffer, gAudio_buffer_size) )
	{
		return -EFAULT;
	}
	tag = strsep ( tempPtr, "=" );

	mutex_lock(&gCodec->mutex);
	if ( strcmp ( tag, "PLAY_CAPABLE" ) == 0 )
	{
		value = strsep ( tempPtr, "=" );
		sscanf ( value, "%d", &uiAudio_data->play_capable );
		playVolume(gCodec->dai);
	}
	if ( strcmp ( tag, "REC_CAPABLE" ) == 0 )
	{
		value = strsep ( tempPtr, "=" );
		sscanf ( value, "%d", &uiAudio_data->rec_capable );
		recBoostLevel(gCodec->dai);
	}
	if ( strcmp ( tag, "REC_BOOST_CAPABLE" ) == 0 )
	{
		value = strsep ( tempPtr, "=" );
		sscanf ( value, "%d", &uiAudio_data->rec_boost_capable );
		//reset level if changing boost capable
		recBoostLevel(gCodec->dai);
	}
	if ( strcmp ( tag, "AEC_CAPABLE" ) == 0 )
	{
		value = strsep ( tempPtr, "=" );
		sscanf ( value, "%d", &uiAudio_data->aec_capable );
	}
	if ( strcmp ( tag, "PLAY_AGC_CAPABLE" ) == 0 )
	{
		value = strsep ( tempPtr, "=" );
		sscanf ( value, "%d", &uiAudio_data->play_agc_capable );
		ui_agcPlayEnable(gCodec->dai, uiAudio_data->play_agc_capable);
	}
	if ( strcmp ( tag, "REC_AGC_CAPABLE" ) == 0 )
	{
		value = strsep ( tempPtr, "=" );
		sscanf ( value, "%d", &uiAudio_data->rec_agc_capable );
		ui_agcRecEnable(gCodec->dai, uiAudio_data->rec_agc_capable);
	}
	if ( strcmp ( tag, "PLAY_VOL" ) == 0 )
	{
		value = strsep ( tempPtr, "=" );
		sscanf ( value, "%d", &uiAudio_data->play_vol );
		if(uiAudio_data->play_vol > PLAY_VOL_MAX)
			uiAudio_data->play_vol = PLAY_VOL_MAX;
		else if(uiAudio_data->play_vol < PLAY_VOL_MIN)
			uiAudio_data->play_vol = PLAY_VOL_MIN;
		playVolume(gCodec->dai);
	}
	if ( strcmp ( tag, "REC_BOOST_LEVEL" ) == 0 )
	{
		value = strsep ( tempPtr, "=" );
		sscanf ( value, "%d", &uiAudio_data->rec_boost_level );
		if(uiAudio_data->rec_boost_level > BOOST_MAX)
			uiAudio_data->rec_boost_level = BOOST_MAX;
		else if(uiAudio_data->rec_boost_level < BOOST_MIN)
			uiAudio_data->rec_boost_level = BOOST_MIN;
		recBoostLevel(gCodec->dai);
	}
	if ( strcmp ( tag, "PLAY_MUTE" ) == 0 )
	{
		value = strsep ( tempPtr, "=" );
		sscanf ( value, "%d", &uiAudio_data->play_mute );
		playVolume(gCodec->dai);
	}
	if ( strcmp ( tag, "REC_MUTE" ) == 0 )
	{
		value = strsep ( tempPtr, "=" );
		sscanf ( value, "%d", &uiAudio_data->rec_mute );
		recBoostLevel(gCodec->dai);
	}

	/* function below is for DEBUG only 
	* this function allows you to write a byte to any of the audio devices cntrol registers
	* set the register with: DBG_WRITE_REG=<reg offset>
	* write the value with: DBG_WRITE_VAL=<byte value>
	*
	*/
	if ( strcmp ( tag, "DBG_WRITE_REG" ) == 0 )
	{
		value = strsep ( tempPtr, "=" );
		sscanf ( value, "%d", &uiAudio_data->dbg_write_reg );
		regVal=aic32x4_read(gCodec, uiAudio_data->dbg_write_reg);
		printk(KERN_ERR "DEBUG reg set is: 0x%x : val=0x%x\n", uiAudio_data->dbg_write_reg, regVal);

	}
	if ( strcmp ( tag, "DBG_WRITE_VAL" ) == 0 )
	{
		value = strsep ( tempPtr, "=" );
		sscanf ( value, "%x", &uiAudio_data->dbg_write_val );
		aic32x4_write(gCodec, uiAudio_data->dbg_write_reg, uiAudio_data->dbg_write_val);
		regVal=aic32x4_read(gCodec, uiAudio_data->dbg_write_reg);
		printk(KERN_ERR "DEBUG reg set is: 0x%x : val=0x%x\n", uiAudio_data->dbg_write_reg, regVal);
	}
	mutex_unlock(&gCodec->mutex);
	return gAudio_buffer_size;
}


/*
 * This function decides whether to allow an operation
 * (return zero) or not allow it (return a non-zero
 * which indicates why it is not allowed).
 *
 * The operation can be one of the following values:
 * 0 - Execute (run the "file" - meaningless in our case)
 * 2 - Write (input to the kernel module)
 * 4 - Read (output from the kernel module)
 *
 * This is the real function that checks file
 * permissions. The permissions returned by ls -l are
 * for referece only, and can be overridden here.
 */

static int module_permission(struct inode *inode, int op, struct nameidata *foo)
{
//  if ( op == 2 ) // no writes
//  {
//    return -EACCES;
//  }

  return 0;
}

/*
 * The file is opened - we don't really care about
 * that, but it does mean we need to increment the
 * module's reference count.
 */
int procfs_audio_open(struct inode *inode, struct file *file)
{
	try_module_get(THIS_MODULE);
	return 0;
}

/*
 * The file is closed - again, interesting only because
 * of the reference count.
 */
int procfs_audio_close(struct inode *inode, struct file *file)
{
	module_put(THIS_MODULE);
	return 0;		/* success */
}

static struct file_operations File_Ops_UiAudio_File = {
	.read 	 = procfs_audio_read,
	.write 	 = procfs_audio_write,
	.open 	 = procfs_audio_open,
	.release = procfs_audio_close,
};

/*
 * Inode operations for our proc file. We need it so
 * we'll have some place to specify the file operations
 * structure we want to use, and the function we use for
 * permissions. It's also possible to specify functions
 * to be called for anything else which could be done to
 * an inode (although we don't bother, we just put
 * NULL).
 */

static struct inode_operations Inode_Ops_File = {
	.permission = module_permission,	/* check for permissions */
};


int uiAudioInit(struct snd_soc_codec *codec)
{
	int error=0;

	gCodec=codec;  //set 

	uiAudio_data = kzalloc(sizeof(struct uiAudio), GFP_KERNEL);
        if(uiAudio_data == NULL)
                return -ENOMEM;

	/* initialize Crestron UI audio data */	
	uiAudioDataDefInit(codec->dai);

 	/* create the /proc file for uiAudio file */
	UiAudio_File = create_proc_entry(UIAUDIO_FILENAME, 0644, NULL);
	if (UiAudio_File == NULL){
		printk(KERN_ALERT "Error: Could not initialize /proc/%s\n",
		       UIAUDIO_FILENAME);
		return -ENOMEM;
	}
	else
	{
	  //UiAudio_File->owner = THIS_MODULE;
	  UiAudio_File->proc_iops = &Inode_Ops_File;
	  UiAudio_File->proc_fops = &File_Ops_UiAudio_File;
	  UiAudio_File->mode = S_IFREG | S_IRUGO | S_IWUSR;
	  UiAudio_File->data = codec;
	  UiAudio_File->uid = 0;
	  UiAudio_File->gid = 0;
	  UiAudio_File->size = 80;
       }

	return error;
}

